#rem
BatteryMonitor Source Code

This source code was created by Wayne Getchell of Sagacitic Solutions, amateur 
radio callsign VE3CZO.  It is meant to be used only with BatteryMonitor hardware.
Do not copy or publish it without permission and citation. If asked, permission is
easily obtained for not for profit use.  Send your request to Wayne at 
getch@sagacitic.com. If you want to use any part or all of it for profit lets talk.

HARDWARE
BatteryMonitor main board versions 0.4 and 1.0
  
SOFTWARE REVISION HISTORY
  V0.0a Initial coding
  V0.1a Added Settimer loop for more consistent time & time tick cal routine
  V0.1c Added peak current and minimum voltage capture
  V0.2a Added watt hours
  V0.3a Current scalingfactor + Sf set routine
  V0.3e Added batteries 3cell SLA 3cell LiPo clean up autorange when scaling
  V0.3f Improves transient response via two GainRange changes per cycle &
		  Faster response C15=2.2uf from 10. C18 added to insure wake on power-up
		  Sw1 & Sw2 human i/f corrections & timing tweaks.
  V0.3g Optimzed transient response positioning 2nd AutoGainRange + bug fixes
  V0.3h Changed current display resolution keeping x.xxA from end of range 1
		  to 9.96A.  Note in range 2 readings increment in 40mA steps.
  V1.0a Initial release menu tidyup ILpeak & Vmin speeded up only one meas/cycle
        Improved LoadPower reading accuracy at low currents 
  V1.0b Improved AutoGainRange, improve Watthour calculation & display resoultion
  V1.0c Improved LoadPower calculation & display resolution.  Introduce RoundIt subroutine
  V1.0d Improve Ah & Wh routine adding x.xx Ah/Wh.  Increase code space by eliminating some 
		  subroutines that are only called once. Fixed AutoGainRange bug that caused
		  wrong range on restart
  V1.0e Modified Hibernate so user doesn't have to push a button to resume from
        hibernation at startup, happens automatically. Removed NextGainRange variable
		  Roll over Ah Wh if over 9,999 and time if over 99hrs.ILraw reassigned fromW8
		  which was reused to w9 exclusive
  V1.0f Gain400 changed to Gain160 for 10mA/bit 1to10A range. Needs R7,R8 value change
  V1.0g Added WaitForSwitch timeout; improved clock minute calc accuracy & tidyup

PROGRAM DOWNLOAD PROCEDURE
  There is no reset function on the 20X2 & this unit employs a soft power switch
  To download a program to the 20X2 if the unit is off and connected to a power 
  source skip steps 1 and 2
    1) Connect BatteryMonitor to a power source 
    2) If the unit turns on then turn it off by momentarily pressing Sw2 
    3) Start the download process on the PC 
    4) When the 'Connecting to hardware...' window appears push S2,and keep it
       held down until programming is complete
#endrem

Start: 
#picaxe 20x2		'directive informs compiler to use 20X2 hardware
'#######################EEPROM SECTION##########################################
'The 20X2 internal EEPROM is used to store battery voltage vs capacity data.
'Up to 8 different battery data sets can be stored in the 256bytes of EEPROM.
'Each entry uses 32bytes.
'
'BatteryMonitor uses a look-up table to turn Vsource into % remaining battery capacity 
'During initialization scratchpad memory locations are populated with the lookup
'table as outlined in the LoadBatteryCap subroutine.
'The % capacity can then be found by coding	get vbat,var1.
'
'The storage format uses a user friendly header of 15 bytes so that the 
'battery can can be easily identified by the user. All 15 bytes must be programmed.
'Use ASCII char " " if there is no need to use all characters.
'The next bytes are used to store the battery voltage vs capacity thresholds in 
'three digit format where xxx = xx.xV as found through characterization. 
'Voltage entries are from low to high and represent 0,2,5,10,20,30,50,60,70,80,90% 
'battery capacity. 
'If not all battery entries are used the first two bytes of the first unused 
'entry must contain $ff,$ff to act as a marker for the menu program.
'20X2 memory limits the battery voltage range to 128bytes, 5.3V to 17.9V
'           2%  5%  10% 20% 30% 40% 50% 60% 70% 80% 90% 99%
'Battery0
EEPROM $00,("3 Cell LeadAcid")
EEPROM $10,(053,054,055,056,057,058,059,060,061,062,063,064)
'Battery1
EEPROM $20,("6 Cell LeadAcid")
EEPROM $30,(109,110,112,114,116,118,120,121,123,124,125,127)
'Battery2
EEPROM $40,("10 Cell NiMH   ")
EEPROM $50,(105,109,113,116,120,123,124,125,126,127,128,129)
'Battery3
EEPROM $60,("11 Cell NiMH   ")
EEPROM $70,(116,120,125,128,133,135,137,138,139,140,141,142)
'Battery4
EEPROM $80,("3 Cell LiPo    ")
EEPROM $90,(095,104,107,110,113,114,116,117,118,120,121,122)
'Battery5
EEPROM $a0,("4 Cell LiPo    ")
EEPROM $b0,(127,138,144,147,150,152,154,156,158,159,161,162)
'Battery6
EEPROM $c0,($ff,$ff)'marker so menu cycles back to first entry
EEPROM $d0,(0)
'Battery7
EEPROM $e0,(0)
EEPROM $f0,(0)
'####End EEPROM Data Load####

'############################# DEFINE VARIABLES #######################
variables:'used by BatteryMonitor 
   symbol BVD=c.4			'switched voltage divider on the Source
 	symbol Gain160=b.2	'Current to voltage amplifer 160x gain setting
 	symbol Gain40=b.3		'Current to voltega amplifier 40x gain setting
 	symbol Backlight1=c.5'display backlight low current input=off high=on
 	symbol Backlight2=c.7'display backlight high current input=off high=on
 	symbol wpEEPROM=b.6	'Write Protect 24AA02 lo=write enable
 	symbol shdn=c.0		'regulator shutdown low=shutdown		'
 	symbol Sw1=pinc.6		'menu switch 1 to 20X2 leg 4 -display line1
 	symbol Sw2=pinb.1		'menu switch 2 to 20X2 leg 10 -display line2
  	
   symbol GainRange=b0	'tracks U1 gain 0=1600, 1=400, 2=40
   symbol var1=b1			'generic variable leased by subroutines
   symbol var2=b2			'generic variable leased by subroutines
   symbol var3=b3			'generic variable leased by subroutines
   symbol unit=b4			'used to parse numbers for display units
   symbol ten=b5			'used to parse numbers for display tens
   symbol hundred=b6		'used to parse numbers for display hundreds
   symbol thousand=b7	'used to parse numbers for display thousands
   symbol tenthousand=b8'used to parse number for display
  	symbol Backlight=b9	'Backlight level   
 	symbol Vsource=b10	'stores battery voltage
 	symbol VsourceMin=b11'captures minimum battery voltage   
   symbol Battery=b12	'stores the battery entry number
   symbol BattPointer=b13'used to select battery REUSED as SetBattLO
	symbol sbattlo=b13	'stored battery entry low value
	symbol sbatthi=b14 	'stored battery entry hi value
	symbol InnerLoopLo=b15'used to write battery level info to scratchpad
	symbol InnerLoopHi=b16'used to write battery level info to scratchpad
  	symbol HibernateState=b17'Flag to determine if device is returning from hibernation
   symbol ILraw=w9		'store raw load current A-D in sys var coparitor results
 	symbol varw10=w10		'Word generic variable leased by subroutines
  	symbol varw11=w11		'Word generic variable leased by subroutines
  	symbol varw12=w12		'Word generic variable leased by subroutines
  	symbol WaitForSwitchTimeout=b26'Counter defines WaitForSwitch timeout interval
	'b27 not used reserved for offset and sign if needed
   symbol MainMenuItem=b28'keeps track of main menu current location
 	symbol BattCap=b29	'stores % battery capacity
	symbol AhWh=b30		'Flag slelects either Ah or Wh for display 1=Wh	
 	symbol Hours=b31		'keeps track of hours	
 	symbol clock=w16		'keeps track of elapsed time in 100ms increments up to 36000
 	symbol Tick=w17		'used to define length of timing element
  	symbol nAh=w18			'accumulates battery energy use basic unit
  	symbol mAh=w19			'accumulates battery energy use
  	symbol Ah=w20			'accumulates battery energy use 
  	symbol uWh=w21			'accumulates battery power basic unit
  	symbol mWh=w22			'accumulates battery power use in mWh
  	symbol Wh=w23			'accumulates battery power use in Watt hours
  	symbol Watts=w24		'stores calculated value for power being used
 	symbol Sf=w25			'Scaling factor used to correct IL reading  	
  	symbol IL=w26			'stores load current
 	symbol ILmax=w27		'maximum current captured during session
 
Initialize:
  	setfreq m4 'optimize power by choosing lowest frequency that will execute 10k samples/sec
  	'Note at 4MHz the pause statements are 2x command eg. a 5ms command is actually 10ms long.
  	high shdn,wpEEPROM,BVD,Backlight2,Gain40 'keep unit powered & 
  	'configure outputs, start with lowest GainRange Av=40 (arbitrary decision)
  	pause 100'time for supply to settle before display initializes
  	hi2csetup i2cmaster,$7C,i2cfast_8,i2cbyte  'configure I2C display
  	hi2cout ($00,$39,$14,$70,$5e,$6b,$0c,$06,$01) 'initialize display
  	gosub SplashScreen
	'write up and down arrow special chars to the display CGRAM
  	hi2cout ($00,$38,$40)'set instruction table to low to write characters
  	hi2cout ($40,$04,$0e,$15,$04,$04,$04,$04,$00,$04,$04,$04,$04,$15,$0e,$04,$00)'write chars
  	hi2cout ($00,$39)	'set instruction table back to hi
   adcsetup=%0100000011000000'removes digital input from ADC converters used
  	pokesfr $22,4	'for C.0 fw - issue after adcsetup to fix V+ reference bug
  	'##### Load Configuraiton Data from External EEPROM ########
  	'                                        clock   nAH     mAh     Ah      uWh     mWh     Wh      
  	hi2cin [$a0],$00,(HibernateState,Hours,b32,b33,b36,b37,b38,b39,b40,b41,b42,b43,b44,b45,b46,b47)  
	'					                  ILmax   Tick     Sf
	hi2cin [$a0],$10,(AhWh,VsourceMin,b54,b55,b34,b35,b50,b51,Battery,Backlight)
	gosub SetBacklightLevel
	if Tick<54300 or Tick>65534 then'outside +/-10% of nominal so use a default value.
		Tick=60400
	endif
	if Sf<900 or Sf>1100 then 'if stored scaling factor is >10% error then use default
		Sf=1000
	endif
  	gosub LoadBatteryCap
  	pause 1000 'leave some time to read the spash screen
	if HibernateState=0 then 
			gosub StartNewReadings 'clear Ah Wh Time data
		else 'hibernationState=1 so device is resuming from Hibernation
  			hi2cout[$7c],($80,$80,$40," Resuming from  ")
  			hi2cout($80,$c0,$40," Hibernation    ")
  			pause 2000 			
  	endif
	GainRange=2 'as defined by Gain160 Gain40 settings
	gosub gainAutoRange'prevents incorrect peak capture on initial cycle
	pause 80'wait a time close to half a measuring cycle before proceeding	
	gosub gainAutorange'second autorange first reading if 2 range changes needed.	
	pause 80'wait a time close to half a measuring cycle before proceeding	
  	
'##########  INITIALIZATION COMPLETE....START MEASURING  #########################
  
 Measure: 'main measuring routine
 	hi2cout [$7C],($80,$01)'clear the display
 	ContinueMeasuring:
 	Timer=$ffff 'preloads timer to overflow in 1 settimer period
  	settimer Tick 'starts timer with period=Tick total loop time target=360ms
  	SetIntFlags %10000000, %10000000 'arm settimer interrupts
  	do while Sw1=0 and Sw2=0 'main measuring loop
 		gosub SourceVoltage
 		gosub LoadCurrent
 		gosub GainAutoRange 'Isense range placed here for max settling time
 		if Vsource<VsourceMin then'records min voltage if gain range hasn't just changed
			VsourceMin=Vsource
		endif
 		if ILmax<IL then 'save maximum peak current if gain range hasn't changed
			ILmax=IL
		endif	
 		gosub LoadPower
 		gosub AmpWattHours
 		gosub ElapsedTime
 		gosub BattCapLeft
 		inc clock
 		gosub GainAutoRange'autorange again in case 2 range changes needed this cycle
 		'high Backlight1,Backlight2 'used only for development. Provides a loop timing test
 		'if loop time is sufficient backlight will blink regularly each cycle
		pause 1000'pause longer than the Settimer period interrupt will occur here 
 	loop
  	settimer OFF'get rid of potential interrupts	
 	setintflags OFF 'disarm interrupts
 	if Sw1=1 then 'check switches Sw1<2sec=Utilites >2sec Hibernate Sw2>2sec=PwrOff
 			for var1=0 to 9
 				pause 70
 				if Sw1=0 then goto Utilities
 			next var1
 			goto GoHibernate
 		else 'Sw2 must be 1
 			for var1=0 to 9
 				pause 70
 					if Sw2=0 then goto ContinueMeasuring
 			next var1
 			if HibernateState=1 then
 				goto GoHibernate
 			else	
 				goto PwrOff
 			endif
 	endif
end

GainAutoRange:'finds best range for load current measurements
  	'current measurements are divided into 3 ranges
 	'range 0 Gain40=0 Gain160=0 Av=1600 1mA/bit range limit 0-999 0-999mA
  	'range 1 Gain40=0 Gain160=1 Av=160 10mA/bit range 100-999 1.00-9.99A
  	'range 2 Gain40=1 Gain160=0 Av=40 40mA/bit range 250-1023 10.0A-40,9A
  	'may have to use this loop twice to find the right gain
  	readadc10 6,ILraw
   if GainRange=0 and ILraw>999 then'reduce gain as current>1A
      	high Gain160 'change gain to 400
      	low Gain40
      	GainRange=1
   	elseif GainRange=1 and ILraw>999 then'reduce gain as current >10A
      	low Gain160 'change gain to Av=40 
      	high Gain40
      	GainRange=2
      elseif GainRange=2 and ILraw<250 then'increase gain as current <10A
      	high Gain160 'change gain to Av=160
      	low Gain40
      	GainRange=1
   	elseif GainRange=1 and ILraw<100 then'increase gain as current <1A
      	low Gain160 'change gain to Av=1600   
      	low Gain40
      	GainRange=0	
  	endif 
return  
 
SourceVoltage: 'measures & displays Vsource as xx.xV
	'Leases 
	'var1 stores dig1 of initial multiply for rounding
	'varw10 'stores voltage for ADC and presentation
	readadc10 7,varw10
	varw10=varw10*24 'with a 6:1 voltage divider and 4.096Vref each A/D bit is 24mV (4.096/1024*6)
	'now round up
	var1=varw10 dig 1 'store voltage reading second least significant bit in var1
	Vsource=varw10/100	'Vsource is now in form xxx
	if var1>4 then 
			Vsource=Vsource+1 'round up Vsource if Var1=>5
	endif
	varw10=Vsource 
	gosub Decimal'returns data for display
	poke 60,hundred,ten,unit 'put battery voltage in displayable format is SFR locations
return
	
LoadCurrent: 'Load current autoranged and displayed as xxxmA, x.xxA or xx.xA
'Leases
'Var1 used in calculations to flag rounding
'Var2 flags Sf>=1
'Varw11 stores intermediate scaling calculation number
'Vwrw12 stores intermediate scaling calculation number
   readadc10 6,ILraw 'take a new reading
  	'now scale the reading
	if Sf>=1000 then'scale factor is greater than 1 
			var2=1
			varw12=Sf-1000'varw12 now holds modified Sf
		else
			var2=0
			varw12=Sf
	endif
  	varw11= varw12 dig 0 * ILraw/20
  	varw11= varw12 dig 1 * ILraw/2 + varw11
  	varw11= varw12 dig 2 * ILraw * 5 + varw11
  	var1=varw11 dig 1
  	varw12=varw11//50
  	varw11=varw11/50
  	if varw12>=25 then 'round the result to three digits
   	inc varw11
  	endif
  	if var2=1 then 'scale factor is greater than 1
  		ILraw=ILraw+varw11
  		else
  		ILraw=varw11
  	endif
 'end scaling factor calculation
 'now calculate IL from scaled ILraw.  Don't round here as watts calculations
 'need unrounded numbers for best accuracy. 
   if GainRange=0 then
  			IL=ILRaw
  		elseif GainRange=1 then
   		IL=ILraw*10
      elseif GainRange=2 then  	
   		IL=ILraw*40 	
  	endif
  	varw10=IL
  	gosub RoundIt
  	gosub Decimal 'now prepare to display the results
	if IL<1000 and GainRange=0 then
       	poke 71,hundred,ten,unit,"m"
		elseif IL<10000 and GainRange<2 then
      	poke 71,thousand,".",hundred,ten
		else
       	poke 71,tenthousand,thousand,".",hundred
	endif
return
     	
LoadPower: 'calculates and displays load power in x.xx W,xx.xW, or xxxxW
	'Leases
	'varw10 -takes reading to display for parsing  & ASCII conversion
	'varw12 -stores IL/10 to prevent mult. overflow
  	'var1 - used for rounding
  	'rem Vsource max=246 ILmax=40920+10%=45012
 	if IL<267 then 'max=6.051W
 			varw10=IL*Vsource
 			gosub RoundIt
 			gosub Decimal
 			poke 63,tenthousand,".",thousand,hundred 'display x.xxW
 		elseif IL >=267 and IL<1331 then 'max=32.7W
 			varw10=IL/5*Vsource/2
 			gosub RoundIt			
 			gosub Decimal
 			if Varw10<10000 then
 					poke 63,thousand,".",hundred,ten'display x.xxW to 9.99W
 				else
 					poke 63,tenthousand,thousand,".",hundred'display xx.xW to32.7W
 			endif
 		elseif IL>=1331 and IL<=26600 then 'max=130W to 5321 and 654W
 			if IL<5321 then
 					varw10=IL/20*Vsource/5
 				else
 					varw10=IL/100*Vsource
 			endif
 			gosub RoundIt
 			gosub Decimal
 			if Varw10<10000 then
 					poke 63,thousand,hundred,".",ten
 				else
 					poke 63," ",tenthousand,thousand,hundred
 			endif
 		else if IL>26600 then'max=999W
 			varw10=IL/200*Vsource/5
 			gosub RoundIt
 			gosub Decimal
 			if Varw10<10000 then
 					poke 63," ",thousand,hundred,ten
 				else
 				poke 63,tenthousand,thousand,hundred,ten
 			endif
 	endif 	
return

ElapsedTime:'calculates and displays elapsed time in HH:mm format
  	'clock is stored in tick increments up to 10000 then Hours is incremented.
'Leases
'varw10 - clock minutes and total time calculations
  	if clock>10000 then
  		inc hours
  		if hours >99 then'reset hours to zero if >99
  			hours=0
  		endif
  		clock=clock-10000
  	endif
  	varw10=clock*6/1000 'find minutes
  	varw10=hours*100+varw10 'add hours to the number displayed
  	gosub Decimal
  	if varw10<1000 then
   		thousand=" " 'blank out tens of hours rather than cram display
  	endif
  	poke 67,thousand,hundred,ten,unit
return

BattCapLeft: 'shows approximate remaining battery capacity as xx%
	if Vsource>179 then 'limit Vsource to stored number limits
			Vsource=179
		elseif Vsource<52 then 
			Vsource=52	
	endif
	get Vsource,Varw10
	gosub Decimal
  	poke 80,ten,unit
return

#rem
'Use this code to display Amplifier range in place of battery capacity
'but don't forget to remark out BattCapLeft subroutine.
BattCapLeft:
varw10=GainRange
gosub Decimal
poke 80,"0",unit
return
#endrem

AmpWattHours:'calculates and displays battery use in Amp hours or Watt hours.
	'displayed as xxxm, x.xx, xx.x, or xxx Ah or Wh
	'Calculations are based on collecting 10k samples per hour (360ms per sample or
	'about 2.8 samples per second)
	'
	'##########################  BEGIN AMP HOUR CALCULATION  ########################
	'rem IL then ranges between 0 and 40,960.  Could be as high as 45012 if Ilraw correction -10%
	'Each measuring cycle accumulates amp hours as IL value + nAh *10^-7Ah
	'Reading interval is 10^-4 (10k samples per hour).  
	'Unit reading is lowest current reading*ReadingInterval=10^-3A*10^-4hr=10^-7Ah
	'Smallest reading is 1mA in 1/10000hr = 10^-7Ah so every 10,000=1mAh.
	'and 1000mAh=1Ah 
  	nAh=IL+nAh 'accumulate battery use
   if nAh>=10000 then 'if over 1mAh
  		var1=nAh dig 4 'large IL can accumulate more than 10k every measuring cycle so find out.
  		varw11=var1*10000
  		mAh=mAh+var1
  		nAh=nAh-varw11 'adjust nAh accumulator to account for mAh
  	endif
  	if mAh >= 1000 then 'accumulate Amp hours and adjust the mAh register
  			Ah=Ah+1
  			If Ah>9999 then 'roll over Ah if value gets to 10k.
  				Ah=0
  			endif
  			mAh=mAh-1000
  	endif
   '########################  BEGIN WATT HOUR CALCULATION  ########################
   'IL is saved in mA and can range from 0 to 40,920 +10% = 45,012
   'Vsource ranges from 52 (5.2V) to 246 (24.6V)
   'Reading interval is 10^-4 (10k samples per hour).  
	'Unit incrment is IL*Vsource*ReadingInterval=10^-3A*10^-1V*10^-4hr=10^-8 Wh
	'Smallest reading is 1mA*0.1V in 1/10000hr = 10^-8Ah so every 100,000=1mWh..
   'Multiplying Vsource with Il to get 10^-4Wh (base unit 0.1V*10^-3A*10^-4h=10^-8Wh) 
   'can easily overflow the processor's 65535 limit.  So this calculation is broken into four
   'ranges to prevent overflow while minimizing rounding errors. .
   'Both base watt hour accumulation unit named uWh and the current IL are divided
   'by a factor in each range that optimizes the resolution while preventing the product
   'from overflowing.
   'rem max Vsource is 246
   'rem symbol uWh is actually 10^-8 mWh  NOT 10^-6 and the symbol mWh is 10^-4 Wh NOT 10^-3 Wh
   	if IL<226 then 'first range for small values of IL.  In this range IL is not divided. 
   	uWh=IL*Vsource+uWh'uWh max =IL*Vsource+10000=225*246+10000=65360
  		var1=uWh dig 4 'large Vsource & IL within range can accumulate more than 10k uWh every cycle.
  		varw11=var1*10000
  		mWh=mWh+var1
  		uWh=uWh-varw11
      elseif IL>=226 and IL<=1290 then 'second range Wh and IL are divided by 5
   		uWh=uWh/5	'change uWh to a value suitable for this range (overflow vs accuracy)
   		uWh=IL/5*Vsource+uWh
   		mWh=uWh/2000+mWh 'as IL uWh were divided by 5 only need to * by 2000 to =10k 
   		uWh=uWh//2000*5'before leaving determine remainder uWh and re-normalize
   	elseif IL>=1292 and IL<=5286 then 'third range Wh and IL are divided by 20
   		uWh=uWh/20 'change uWh to a value suitagle for this range (overflow vs accuracy)
   		uWh=IL/20*Vsource+uWh '/20 uWh max =IL/20*Vsource+uWh=4396/20*246+10000/20=65444
   		mWh=uWh/500+mWh 'calculate mWh
   		uWh=uWh//500*20 'calculate residual
   	elseif IL>=5287 and IL<=26599 then 'fourth range Wh & Il are divided by 100
   		uWh=uWh/100
   		uWh=IL/100*Vsource+uWh '/100 uWh max=IL/100*Vsource+uWh=65290
   		mWh=uWh/100+mWh
   		uWh=uWh//100*100
   	elseif IL>26599 then 'fifth range for very large currents.
   		uWh=uWh/200 
   		uWh=IL/200*Vsource+uWh '/200 uWh max =IL/200*Vsource+uWh=45012/200*246*9999/200+1=55175
   		mWh=uWh/50+mWh
   		uWh=uWh//50*200 'residual
   endif
	if mWh>=10000 then 'if 1000 real mWh (10k in this calculation) accumulated increment Wh & decrement mWh
		Wh=Wh+1
		if Wh>9999 then 'roll over Wh if value gets to 10k
			Wh=0
		endif 
   	mWh=mWh-10000
	endif
	'Now display either Ah or Wh
	if AhWh=0 then 'display Ah
    			if Ah=0 then 'under 1Ah so display in mAh     		
  					varw10=mAh
    				gosub Decimal
      			poke 75,hundred,ten,unit,"m","A"
  				else if Ah<100 then
  					var1=mAh dig 2*10 'parse hundreds of an Ah (don't bother to round up)
  					var2=mAh dig 1
  					varw10=Ah*100+var1+var2 'creates Ah up to 3 digits for display parsing
  					gosub Decimal
  					if varw10<1000 then 'Ah is less than 10
  							poke 75,hundred,".",ten,unit,"A"
  						else
  							poke 75,thousand,hundred,".",ten,"A"
  					endif
  				else 
  					varw10=Ah
  					gosub Decimal
      			if varw10<1000 then
      					poke 75," ",hundred,ten,unit,"A"'supress leading zero
      				else
      					poke 75,thousand,hundred,ten,unit,"A"'max display is 6553Ah
      			endif
      		endif
  	endif
  	if AhWh=1 then'display Wh
  			'Now display mWh or Wh
  			'For display remember Wh to three or four digits if >1k
  			if Wh=0 then
  					varw10=mWh
  					gosub Decimal
  					poke 75,thousand,hundred,ten,"m","W"
  				elseif Wh<100 then 'count is over 1Wh but less than 100
  					var1=mWh dig 3*10'parse for tenths of a Watt hour
  					var2=mWh dig 2'parse for 100'th of a Watt hour
  					varw10=Wh*100+var1+var2'creates Wh with .1Wh resolution for display
  					gosub Decimal
  					if varw10<1000 then
  							poke 75,hundred,".",ten,unit,"W"
  						else
  							poke 75,thousand,hundred,".",ten,"W"
  					endif
  				else 'Wh>100
  					varw10=Wh
  					gosub Decimal
  					if varw10<1000 then 'Wh is less than 1000 so blank leading 0
  					 		poke 75," ",hundred,ten,unit,"W"
  						else 'Wh is over 1000 so display leading digit.
  							poke 75,thousand,hundred,ten,unit,"W"
  					endif
  			endif
  	endif	      
return 	

LoadBatteryCap:'loads a lookup table of battery capacity into scratchpad memory
	'Vsource reading (3 digits) is used as the address, battery capacity in % is the data
	'So to get battery capacity code -  get vbat,var1 where var1 now holds batt cap.
	'capacity is displayed as a two digit number = xx%.
	'Uses all 128 bytes of scratchpad memory.
	
	'First all readings for the selected battery are assembled in FSR locations 65-76
	'add 053 as the first entry and 179 as the last to the FSR list. This represents
	'5.3V minimum and 17.9V maximum, the range limited by 128 bytes of available memory
	BattPointer=2*Battery*$10+$10'set pointer to start of battery entry numeric data
	for var1=65 to 76
		read BattPointer,var2
		poke var1,var2
		inc BattPointer 'get next entry
	next
	poke 64,53	'store minimum battery voltage reading
	poke 77,179 'store max battery voltage reading 
	'now associate the battery readings to a lookup table that will return battery capacity
	'To do this populate the scratchpad so that when an address is used that equals
	'Vsource the entry returns the battery capacity as defined by the key percentage
	'markers in the list stored in the FSR. 
	'with entries such that using any voltage between min and max as a 'get' address
	'will return the percentage as prescribed by the battery entry data
	'because the 20X2 only has 128byte scratchpad  entries from 128 to 179 
	'will appear in scratchpad 0 to 51 52=0
	'But addressing >128 loop the counter & return the correct battery capacity.	
	InnerLoopLo=64
	InnerLoopHi=65
	for var1=0 to 12
		lookup var1,(0,2,5,10,20,30,40,50,60,70,80,90,99),BattCap'now contains %cap
		peek InnerLoopLo,sbattlo
		peek InnerLoopHi,sbatthi
		for var3=sbattlo to sbatthi
			put var3,BattCap
		next
		inc InnerLoopLo
		inc InnerLoopHi
	next
return

  '###################### UTILITIES SECTION #################################
Utilities:
  'displays main title on the display line 1 and menu items on line 2
  'pressing S1 increments menu choice, S2 selects
  'While in utilities menus time stands still, Ah & Wh don't accumulate.
	'Leases
  	'None
  	MainMenuItem=0	'keeps track of menu items; init to 0
  	MainMenuInnerMenuLoop:
  	if MainMenuItem>7 then
  		MainMenuItem=0
  	endif
  	gosub ClearDisplay
  	hi2cout [$7c],($80,$80,$40,$01,"Scroll BM Utils")'write MainMenu header
  	hi2cout ($80,$c0,$40,$7f)'set display for start for most menu items to save code 
  	on MainMenuItem gosub MenuMinAndPeak,MenuClearReadings,MenuChooseAhWh,MenuSetBacklight,MenuSelectBattery,MenuGoHibernate,MenuPwrOff,MenuCalibrate
  	pause 400 'wait a bit so display can be viewed
  	gosub WaitForSwitch
  	if Sw1=1 then
  			pause 100 'bit of time to scroll menu if button held down
   		inc MainMenuItem
    	elseif Sw2=1 then
    		pause 400 'get finger off button
    		gosub ClearDisplay 
      	on MainMenuItem gosub MinAndPeak,ClearReadings,ChooseAhWh,SetBacklight,SelectBattery,GoHibernate,PwrOff,Calibrate
      	pause 400 'get finger off button    		
      	goto EndMainMenuLoop 'returned from subroutine so go back to measuring
  		else
  			goto EndMainMenuLoop'Neither switch was pushed and WaitForSwitch timed out so resume measuring
  	endif
   goto MainMenuInnerMenuLoop
  	EndMainMenuLoop:
  	goto Measure

MenuMinAndPeak: 'Restarts display
  	hi2cout ($40,"Show Vmin & ILp")
return
MenuClearReadings: 'Restarts display
  	hi2cout ($40,"Clear Readings")
return
MenuChooseAhWh: 'Selects either Amp hours or Watt hours for display
  	hi2cout ($40,"Choose Ah or Wh")
return
MenuSetBacklight: 'sets backlight level
  	hi2cout ($40,"Set Backlight")
return
MenuSelectBattery:'Battery Type selection submenu
  	hi2cout ($40,"Select Battery")
return
MenuGoHibernate: 'Hibernation subroutine
  	hi2cout ($40,"Hibernate")
return
MenuPwrOff:
  	hi2cout ($40,"Power Off")
return
MenuCalibrate: 'Calibration submenu
  	hi2cout ($40,"Calibrate")
return 	

MinAndPeak: 'Displays minimum voltage and peak current
	hi2cout ($80,$80,$40,$7f,"Y Clr Vmin & Ipk?")
	varw10=VsourceMin
	gosub Decimal'returns data for display
	hi2cout($80,$c0,$40,$7f,"N ",hundred,ten,".",unit,"Vm")
	varw10=ILmax
	gosub Decimal
	  	if ILmax>10000 then
	        	hi2cout($80,$ca,$40,tenthousand,thousand,".",hundred,"Ap")
			elseif ILmax<1000 then 	
        		hi2cout($80,$ca,$40,hundred,ten,unit,"mAp")
     		else
       		hi2cout($80,$ca,$40,thousand,".",hundred,ten,"Ap")
      endif	
	gosub WaitForSwitch
	if Sw1=1 then 'reset peak values
		VsourceMin=255
		ILmax=0
	endif
return

ClearReadings:'clears all stored readings resetting to zero & resets hibernation state
	hi2cout ($80,$80,$40,$7f,"Y  Clear")
	hi2cout ($80,$c0,$40,$7f,"N  Readings?")
	pause 400 'finger off button
	gosub WaitForSwitch
	if Sw1=1 then 'begin the process of clearing all readings
			pause 400 'finger off button
			if HibernateState=1 then 'only wirte the EEPROM if necessary
				HibernateState=0
				low wpEEPROM 'enable EEPROM write
				hi2cout[$a0],$00,(HibernateState)'and save int in EEPROM
				pause 10'EEPROM write cycle time
				high wpEEPROM
			endif
			gosub StartNewReadings
		elseif Sw2=1 then
			pause 400 'finger off button
	endif		
return

ChooseAhWh:'Menu item selects display format Amp hours or Watt hours
	hi2cout ($80,$80,$40,$7f,"Ah  Sel Display")
	hi2cout ($80,$c0,$40,$7f,"Wh  Format")
	gosub WaitForSwitch
	if Sw1=1 then
			AhWh=0
		elseif Sw2=1 then
			AhWh=1
	endif
	low wpEEPROM 'enable EEPROM write
	hi2cout[$a0],$10,(AhWh) 'and save in EEPROM
	pause 200'EEPROM write cycle time & finger off button
	high wpEEPROM	
return

SetBacklight:'sets backlight to one of four levels: off,low,med,hi and saves setting
 	hi2cout ($80,$80,$40,$01," Next BL Level")'set up display heading for this section
	hi2cout ($80,$c0,$40,$7f," Backlight")

	if Backlight=0 then
			hi2cout ($80,$cc,$40,"Off")
		else if Backlight=1 then			
  			hi2cout ($80,$cc,$40,"Low")
		else if Backlight=2 then			
  			hi2cout ($80,$cc,$40,"Med")
		else 'Backlight must be 3			
  			hi2cout ($80,$cc,$40,"Hi ")
  	endif
  	gosub SetBacklightLevel
	gosub WaitForSwitch
	if Sw1=1 then
		pause 400 'finger off switch
		inc Backlight
		if Backlight>3 then
			Backlight=0
		endif
		goto SetBacklight
	elseif Sw2=1 then
		pause 400'finger off switch
		low wpEEPROM 'save backlight setting in EEPROM
		hi2cout[$a0],$19,(Backlight)
		pause 10'wait for write to complete
		high wpEEPROM
	else 'WaitForSwitch timed out so reset the backlight level & continue measuring
		hi2cin [$a0],$19,(Backlight)
		gosub SetBacklightLevel
	endif
	return
  	 	
SetBacklightLevel:	
  	if Backlight=0 then 'backlight is off
  			low Backlight1,Backlight2
  		elseif Backlight=1 then'backlight is on low
  			input Backlight2
  			high Backlight1
  		elseif Backlight=2 then 'backlight is on medium
  			input Backlight1
  			high Backlight2
  		elseif Backlight=3 then 'backlight is on high
  			high Backlight1,backlight2
  	endif		
return

SelectBattery: 'start with battery being used then cycle through battery choices
 	hi2cout ($80,$80,$40,$01," Next Battery")
	BattPointer=2*Battery*$10'set pointer to start of battery entry numeric data
	for var1=$c1 to $cf
		read BattPointer,var2
		hi2cout ($80,$c0,$40,$7f)
		hi2cout ($80,var1,$40,var2)
		inc BattPointer 'get next entry
	next	
	gosub WaitForSwitch
	if Sw1=1 then
			pause 400 'get finger off button
			inc Battery
			if Battery=8 then'only 8 batteries in the list
				Battery=0
			endif
			BattPointer=2*Battery*$10
			read BattPointer,Var2
			inc BattPointer
			read BattPointer,Var3
			if Var2=$ff and Var3=$ff then'beginning of any empty entries uses $ff,$ff as tag
				Battery=0
			endif
			goto SelectBattery
		elseif Sw2=1 then
			pause 400 'get finger off button
			low wpEEPROM 'write enable EEPROM
			hi2cout[$a0],$18,(Battery)
			pause 10 'time for write
			high wpEEPROM'write protect EEPROM
			gosub LoadBatteryCap
		else 'WaitForSwitch timed out so get original battery info from EEPROM
			hi2cin[$a0],$18,(Battery) 'get the stored battery info
	endif
return

GoHibernate:'FlagHibernation save current data in EEPROM and go to sleep
	HibernateState=1
	low wpEEPROM'enable EEPROM write
	'External EEPROM write requirements - 8byte max writes must start on a
	'writepage boundary, $00,$08,$10,$18 etc.
	'                                         Time    nAh     mAh  
	hi2cout [$a0],$00,(HibernateState,Hours,b32,b33,b36,b37,b38,b39)
	pause 10 'write time
	'               Ah     uWh     mWh     Wh
	hi2cout $08,(b40,b41,b42,b43,b44,b45,b46,b47)
	pause 10 'write time
	'                       ILmax
	hi2cout $11,(VsourceMin,b54,b55)
	pause 10 'write time
	high wpEEPROM'disable EEPROM write
'now proceed to PwrOff NB PwrOff must start after GoHibernate

PwrOff:
	gosub ClearDisplay
	if HibernateState=1 then
			hi2cout($80,$82,$40,"Hibernating")
		else
  			hi2cout($80,$86,$40,"Bye")
  	endif
  	hi2cout[$7c],($80,$c6,$40,"Bye")
  	pause 1000'wait a bit for the button to be released or unit will repower
  	low shdn
  	pause 2000 'pause 4 seconds to make sure user finger is off the pushbutton
  	high shdn'just in case it didn't shut down mustn't stay low or Sw1 inoperative
goto Initialize
 	    
Calibrate:'store value for ScaleFactor; calculate and store Tick
#rem
Scale Factor
The scale factor corrects current measurements accounting for errors in Vref voltage,
shunt resistor value, and current to voltage converter gain based on the gain of 
GainRange0. Note that this routine does not calculate the scaling factor, it only 
enables the user to enter and save a calculated scale factor.  
To accurately calculate the scale factor follow these steps:
Apply a load current to the battery monitor that is close to the upper end of 
GainRange0. The current should be in the range of 930 to 990mA with an known 
naccuracy of +/-.25%.  A vriable power supply around 12 to 15 volts and a stable 
power resistive load of about 15 ohms works well to produce the desired current.
Insure BatteryMonitor remains in GainRange0. If it autoranges to a higher range,
reduce the current to the upper range of GainRange 0. 
The load current can be measured using a current meter or a precision shunt resistor
in series with BatteryMonitor and the resistive load. If a resistive shunt is used
use one in the range of 1.0 or 0.1 Ohms with a tolerance of no greater than
0.25%.  The voltage can then be measured across the shunt resistor with an accurate
voltmeter and the load current calculated. 
Note the current reading on the display.  Calculate the scale factor by dividing
the measured load current by the reading on the display. 
Enter this number in the Sf routine, The Sf routine will initially display the
present value for the scale factor. The reading takes the form xxxx where the most
significant digit is either zero or one. For example 1.056 enters as 1056, 0.983
as 0983. To enter a new value push Sw1 next to the up arrow to increase the displayed
reading or Sw2 next to the down arrow to decrease the reading.
The unit accepts any value within a 10% limit.(0900 to 1100).
Because of code space limitations you can enter and store a number outside 
the limit range but on restart the unit will default to a Sf=1.000

Tick
In order to accumulate Ah & Wh accurately the unit must collect 10,000 samples 
per hour 3600/10000=0.360 seconds per cycle. Cycle time is controlled
by a Timer/Settimer interrupt loop. All BatteryMonitor measurements
and calculations are placed within this timed loop. Two activities, servicing the 
loop interrupt and writing the display are outside the timing loop. 
But these activities use a fixed time per cycle as the code is straight line, 
no subroutines or branching. At a 4MHz clock these activities use about 8% of 
the total period so the interrupt driven timer that controls the loop time is adjusted, 
reduced by 8%, to account for the external tasks. 
Timer is set to ffff so it will overflow in one major tick and Setimer is preloaded so 
that it overflows and creates an interrupt in a fixed period.
The processor clock is trimmed to within 1% frequency during manufacture so is quite 
accurate without calibration tweaking.  But the cycle can be made even more accurate 
by adjusting the Settimer preload value by a number of 'ticks'.to account for any 
frequency errors. At 4MHz each 'tick' changes the cycle time by 64us* 1+%time outside 
the loop or 70us.
'Each hour can therefore be adjusted by +/-700mS.  So the best accuracy that can
'be expected is about 2-3 seconds in three hours. 
'Tick will run with a default value if a calibration hasn't been performed.
'A stop watch is required to calibrate the time.
'A reliable power source such as a battery is required to power BatteryMonitor
'A load isn't required for this procedure.
'Invoke the utilities menu by powering the unit and pressing S1
'Navigate to the 'Clear Readings' menu item and select it.
'Press S1 to Clear readings and at the same time start the stopwatch
'Come back about three hours later
'Watch the time display.  Just as the minutes readout changes simultaneously
'push Sw1 and stop the stopwatch.  When Sw1 is pushed the clock stops incrementing
'and the user enters the Utilities menu.  In the Utilities menu navigate to the 
'Calibrate menu then push Sw2 to enter the Time-Tick Cal submenu.  From here
'Enter the time from the stopwatch, first hours, minutes, then seconds.
'Press Sw2 to calculate then save the new value for Tick.  
'Also note this value for future reference.
'It may be necessary to perform this cal procedure a couple times until Tick
'no longer changes by more than a few counts.
#endrem
'Leases
'Var1 stores hours
'var2 stores seconds
'var3 negative flag
'var10 stores difference between internal and external seconds & displays no.
'var11 stores internal elapsed time in seconds
'var12 stores external seconds elapsed time in seconds  		
	hi2cout ($80,$80,$40,$7f," Sf Set")
	hi2cout ($80,$c0,$40,$7f," Time-Tick Cal")
	gosub WaitForswitch
	if Sw1=0 and Sw2=0 then 'WaitForSwitch timed out so go back to measuring 
		goto EndCal
	elseif Sw1=1 then
		goto ScaleFactorSet 
	endif
'Note The WaitForSwitchTimeout is NOT active when in either the Sf Set or Time-Tick Cal routines
	var1=3 'temporary storage for hours
	var2=0 'temporary storage for minutes
	varw11=0 'temporary storage for seconds
	pause 400'get finger off button
	GetHours:
	varw10=var1
	gosub Decimal
	gosub ClearDisplay
	hi2cout ($80,$80,$40,$00," Chg Hrs")
	hi2cout ($80,$c0,$40,$7f," ",unit)
	gosub WaitForSwitch
	if Sw1=1 then 
		pause 200 'finger off button
		inc var1
		if var1>8 then 'arbitrary maximum time interval -8hrs 59 min 59 sec.
			var1=0
		endif
		elseif Sw2=1 then
			pause 400 'get finger off switch 
			goto GetMinutes
	endif
	goto GetHours
	GetMinutes:
	varw10=var2
	gosub Decimal
	hi2cout ($80,$80,$40,$00," Chg Min")
	hi2cout ($80,$c0,$40,$7f," ",ten,unit)
	gosub WaitForSwitch
	if Sw1=1 then 
		pause 100 'increment display time control
			inc var2
			if var2>59 then
				var2=0
			endif
		elseif Sw2=1 then
			pause 400 'get finger off switch
			goto GetSeconds
	endif
	goto GetMinutes		
	GetSeconds:
	varw10=varw11
	gosub Decimal
	hi2cout ($80,$80,$40,$00," Chg Sec")
	hi2cout ($80,$c0,$40,$7f," ",ten,unit)
	gosub WaitForSwitch
	if Sw1=1 then
			pause 100 'increment display time control 
			inc varw11
			if varw11>59 then
				varw11=0
			endif
		elseif Sw2=1 then
			pause 400 'get finger off switch
			goto AskIfTimeCorrect
	endif
	goto GetSeconds
	AskIfTimeCorrect:	
	hi2cout ($80,$80,$40,$7f,"Y Time Ok?")
	hi2cout ($80,$c0,$40,$7f,"N  ")
	varw10=var1 'show hours
	gosub Decimal
	hi2cout ($40,unit,":")
	varw10=var2 'show minutes
	gosub Decimal
	hi2cout($40,ten,unit,":")
	varw10=varw11'show seconds
	gosub Decimal
	hi2cout($40,ten,unit)		
	gosub WaitForSwitch
	if Sw2=1 then
			pause 400 'get finger off switch
			goto GetHours 'Time entry was wrong so try again
	endif
	varw12=var1*3600'stopwatch hours to seconds
	varw12=var2*60+varw11+varw12'recorded stopwatch time in seconds
	varw11=Clock*3'now calculate internal time in seconds as there are
	varw11=Clock*6/10+varw11'10000 samples per hour sec=clock*.36 so 
	var1=varw11 dig 0 'multiply clock by 3 then add remainder 6/10 then 
	varw11=varw11/10'divide by 10 to get value *.36 then round
	if var1>4 then
		inc varw11'now have Clock value in seconds so have to.
	endif				'calculate total time in seconds in next line
	varw11=Hours*3600+varw11'calculates total internal clock time in seconds
	varw10=varw12-varw11'seconds diff between Stopwatch and internal clock
	'positive var10 indicates Stopwatch is faster than internal clock so 
	'Tick should be increased.
	if varw10>32768 then'if number is greater than half it's likely negative so...
		varw10=varw11-varw12'simpler than two's compliment
		var3=1'set flag indicates internal clock is faster than stopwatch
				'so Tick delay should be shortened.
	endif
	'each Tick shifts the internal clock by 0.64 seconds per hour so calculate
	'number of whole ticks to change by multiplying sec./hr *1.563
	'to do this while optimizing error correcting range and resolution
	'multiply seconds error *15.63 then divide by stopwatch time in seconds
	'/36 to get decimal hours *10.
	'This will correct for clock errors in excess of 10% without overflow
	'Note PICAXE devices with internal clocks are trimmed to 1%.
	varw12=varw12/360'total no of hours *10 to optimize resolution
	varw11=varw10*63/100'remainder for 15 63
	varw10=varw10*15+varw11/varw12 'delta sec*15+remainder/timedecimalHrs*10
	if var3=1 then
			varw12=Tick-varw10
		else
			varw12=Tick+varw10
	endif
	varw10=Tick
	gosub Decimal
	hi2cout[$7c],($80,$80,$40,$7f,"Y Save New Tick?")
  	hi2cout($80,$c0,$40,$7f,"N ",tenthousand,thousand,hundred,ten,unit)
  	varw10=varw12
  	gosub Decimal
  	hi2cout($40," ",$7e," ",Tenthousand,thousand,hundred,ten,unit)
  	pause 600 'get finger off switch
  	gosub WaitForSwitch
  	if Sw1=1 then
  		Tick=varw12
		low wpEEPROM'enable EEPROM write
		hi2cout[$a0],$14,(b34,b35) 'save Tick
		pause 10
		high wpEEPROM'disable EEPROM write
		varw10=Tick
		gosub Decimal
		gosub ClearDisplay
		hi2cout[$7c],($80,$83,$40,"Tick=",tenthousand,thousand,hundred,ten,unit)
		pause 2000 'some time to read display
	endif		
EndCal:	
return

ScaleFactorSet:
	varw10=Sf
	gosub Decimal
	gosub ClearDisplay
	hi2cout ($80,$80,$40,$00," Chg Sf")
	hi2cout ($80,$c0,$40,$01," ",thousand,hundred,ten,unit)
	gosub WaitForSwitch
	pause 100	'wait a bit in case both switches are pressed
	if Sw1=1 and Sw2=1 then
			goto WriteSfEEPROM
		else if Sw1=1 then
			inc Sf
		else if Sw2=1 then
			dec Sf
	endif
	goto ScaleFactorSet
	WriteSFEEPROM:
		low wpEEPROM'enable EEPROM write
		hi2cout[$a0],$16,(b50,b51) 'save Sf
		pause 10
		high wpEEPROM'disable EEPROM write
		gosub ClearDisplay
		hi2cout[$7c],($80,$84,$40,"Sf=",thousand,hundred,ten,unit)
		pause 2000 'get finger off button
EndScaleFactorSet:
return

StartNewReadings:'zeros clock and Ah Wh data to start new readings
	clock=0
	Hours=0
	nAh=0
	mAh=0
	Ah=0
	uWh=0
	mWh=0
	Wh=0
	VsourceMin=255
	ILmax=0
return

RoundIt: 'used to round many calculations
'This code rounds numbers for presentation to the display.  It assumes that three
'digits will be displayed.  If the number is less than 10000 then the units digit 
'is rounded, If over 10,000 the the tens digit is rounded. 
'Rounding up occurs if the digit is 5 or more.
'Leases
'var1 - rounding flag
'var10- input to the routine for rounding, output rounded result also is varw10
		if varw10<1000 then 'nothing to round so leave
			goto EndRoundIt
		elseif varw10<10000 then
 	 		var1=varw10 dig 0'round up units to tens for small values
 			if var1>4 then
 				varw10=varw10+10
 			endif
 		else 'variable is over 10,000
 			var1=varw10 dig 1'round up tens to hundreds
 			if var1>4 then
 				varw10=varw10+100
 			endif
 	endif
EndRoundIt:
return
  	
Decimal: 'parses contents of varw10 to digits that can be used directly by the display
  'Leases
  'varw10 used as value to parse 
  	unit=varw10 dig 0 + "0"	'format for display by taking each digit to be displayed
  	ten=varw10 dig 1 + "0"	'and placing it in a variable for use with the hi2cout command
  	hundred=varw10 dig 2 + "0"
  	thousand=varw10 dig 3 + "0"
  	tenthousand=varw10 dig 4 + "0"
return

SplashScreen:'Creates information displayed to the user at power up.
  	hi2cout($40,"BatteryMon V1.0g")'cursor is at home pos so write first line
  	hi2cout($80,$c0,$40,"Source",$7e,$7e,$7e,$7e,$7e,$7e,"Load")'cursor to start of 2nd line & write
return

ClearDisplay:'Erases all displayed data so that the screen is blank.
  	hi2cout [$7C],($80,$01) 'clear display
  	pause 1
return

WaitForSwitch:'Waits for one of two push buttons to be closed and does a cheap 100ms debounce.
  	WaitForSwitchTimeout=0 'reset WaitForSwitch timeout register
  	do while Sw1=0 and Sw2=0 and WaitForSwitchTimeout<200
  	pause 50'really cheap debounce & part of WaitForSwitchTimeout
  	inc WaitForSwitchTimeout' Total time out =50*2*200=20sec.
	loop
return  
 
Interrupt:'special subroutine is entered each time Tick overflows from $ffff.  
	WriteDisplay:'2line x 16 char display updated here. MUST be done OUTSIDE the
					 'Timer Settime loop or code is corrupted by I2C output routine!
	'low Backlight1,Backlight2'a test of loop timing
	peek 60,b1,b2,b3,b4,b5,b6,b7,b20,b21,b22,b23
	hi2cout($80,$80,$40,b1,b2,".",b3,"V ",b4,b5,b6,b7,"W",b20,b21,":",b22,b23)
	peek 71,b1,b2,b3,b4,b5,b6,b7,b8,b20,b21,b22
	hi2cout($80,$c0,$40,b1,b2,b3,b4,"A ",b5,b6,b7,b8,b20,"h ",b21,b22,"%")
	Timer=$ffff'preload timer to flag at next tick
	Settimer Tick	
   SetIntFlags %10000000,%10000000   'Set Timer 0 to interrupt
return